package parser;

/**
 * Factory for E grammar non-terminal objects.
 */
public class EFact extends ATVFactory {
    /**
     * Factory for E1 grammar non-terminals.
     */
    private E1Fact _e1Fact;

    /**
     * Visitor for E1 grammar non-terminals.
     */
    private ITokVisitor _parseE1;

    /**
     * Initializer lambda for this factory.
     */
    private ILambda _initializer = new ILambda() {
        public Object apply(Object param) {
            // change state to no-op
            _initializer = NoOpLambda.Singleton;

            // initialize
            _parseE1 = _e1Fact.makeVisitor();
            return null;
        }
    };

    /**
     * Constructor for the E factory,
     *
     * @param tkz tokenizer to use
     */
    public EFact(ITokenizer tkz) {
        super(tkz);
        _e1Fact = new E1Fact(tkz);
    }

    /**
     * Make a token visitor to parse an E non-terminal.
     *
     * @return token visitor
     */
    public ITokVisitor makeVisitor() {
        initialize();
        return new NumToken.INumVisitor() {
            public Object numCase(NumToken host, Object inp) {
                return new E(host, (E1) nextToken().execute(_parseE1, inp));
            }

            public Object defaultCase(AToken host, Object param) {
                throw new IllegalArgumentException("Wrong token: '" + host + "'");
            }
        };
    }

    /**
     * Make a token visitor that delegates to the given visitor in a chain of responsibility
     *
     * @param successor visitor to serve as successor in the chain
     */
    public ITokVisitor makeChainedVisitor(ITokVisitor successor) {
        initialize();
        return new NumToken.AChainVis(successor) {
            public Object numCase(NumToken host, Object inp) {
                return new E(host, (E1) nextToken().execute(_parseE1, inp));
            }
        };
    }

    /**
     * Make the visitor.
     */
    private void initialize() {
        _initializer.apply(null);
    }

    /**
     * Factory for Empty grammar terminals.
     */
    public class EmptyFact extends ATVFactory {
        /**
         * Constructor for Empty grammar terminals.
         *
         * @param tkz tokenizer to use
         */
        public EmptyFact(ITokenizer tkz) {
            super(tkz);
        }

        /**
         * Make a visitor to parse an empty terminal.
         *
         * @return visitor to parse an empty terminal
         */
        private ITokVisitor makeEmptyVis() {
            return new ITokVisitor() {
                public Object defaultCase(AToken host, Object param) {
                    putBackToken();
                    System.err.println("Pushback");
                    return Empty.Singleton;
                }
            };
        }

        /**
         * Make a token visitor to parse an Empty terminal.
         *
         * @return token visitor
         */
        public ITokVisitor makeVisitor() {
            return makeEmptyVis();
        }

        /**
         * Make a token visitor that delegates to the given visitor in a chain of responsibility.
         *
         * @param successor visitor to serve as successor in the chain
         */
        public ITokVisitor makeChainedVisitor(ITokVisitor successor) {
            return makeEmptyVis();
        }
    }

    /**
     * Factory for E1 grammar non-terminal objects.
     */
    public class E1Fact extends ATVFactory {
        /**
         * Factory for E1a grammar non-terminals.
         */
        private E1aFact _e1aFac;

        /**
         * Factory for Empty grammar terminals.
         */
        private EmptyFact _emptyFac;

        /**
         * Constructor for the E1 factory,
         *
         * @param tkz tokenizer to use
         */
        public E1Fact(ITokenizer tkz) {
            super(tkz);
            _e1aFac = new E1aFact(tkz);
            _emptyFac = new EmptyFact(tkz);

        }

        /**
         * Make a token visitor to parse an E1 non-terminal.
         *
         * @return token visitor
         */
        public ITokVisitor makeVisitor() {
            return _e1aFac.makeChainedVisitor(_emptyFac.makeVisitor());
        }

        /**
         * Make a token visitor that delegates to the given visitor in a chain of responsibility
         *
         * @param successor visitor to serve as successor in the chain
         */
        public ITokVisitor makeChainedVisitor(ITokVisitor successor) {
            return _e1aFac.makeChainedVisitor(_emptyFac.makeChainedVisitor(successor));
        }

        /**
         * Factory for E1a grammar non-terminal objects.
         */
        public class E1aFact extends ATVFactory {
            /**
             * Factory for E grammar non-terminals.
             */
            private EFact _eFact;

            /**
             * Visitor for E grammar non-terminals.
             */
            private ITokVisitor _parseE;

            /**
             * Initializer lambda for this factory.
             */
            private ILambda _initializer = new ILambda() {
                public Object apply(Object param) {
                    // change state to no-op
                    _initializer = NoOpLambda.Singleton;

                    // initialize
                    _parseE = _eFact.makeVisitor();
                    return null;
                }
            };

            /**
             * Constructor for the E1a factory.
             *
             * @param tkz tokenizer to use
             */
            public E1aFact(ITokenizer tkz) {
                super(tkz);
                _eFact = EFact.this;
            }

            /**
             * Make a token visitor to parse an E1a non-terminal.
             *
             * @return token visitor
             */
            public ITokVisitor makeVisitor() {
                initialize();
                return new PlusToken.IPlusVisitor() {
                    public Object plusCase(PlusToken host, Object inp) {
                        return new E1a(host, (E) nextToken().execute(_parseE, inp));
                    }

                    public Object defaultCase(AToken host, Object param) {
                        throw new IllegalArgumentException("Wrong token: '" + host + "'");
                    }
                };
            }

            /**
             * Make a token visitor that delegates to the given visitor in a chain of responsibility
             *
             * @param successor visitor to serve as successor in the chain
             */
            public ITokVisitor makeChainedVisitor(final ITokVisitor successor) {
                initialize();
                return new PlusToken.AChainVis(successor) {
                    public Object plusCase(PlusToken host, Object inp) {
                        return new E1a(host, (E) nextToken().execute(_parseE, inp));
                    }
                };
            }

            /**
             * Make the visitor.
             */
            private void initialize() {
                _initializer.apply(null);
            }
        }
    }
}

